/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.daemon;

import java.net.InetAddress;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jrobin.core.Util;
import org.videolan.vlma.common.IVlData;
import org.videolan.vlma.common.medias.IVlMedia;
import org.videolan.vlma.common.orders.VlOrder;
import org.videolan.vlma.common.programs.IVlProgram;

/**
 * This class is an orders monitoring daemon. It connects to the router and
 * checks that the orders are correctly executed.
 * 
 * @author SylV
 * @version 1.0
 */
public class VlOrderMonitoring {
    
    private static final Logger logger = Logger.getLogger(VlOrderMonitoring.class);

    private IVlData data;
    
    private Thread orderMonitorThread;
    
    private Thread orderMonitorDeamonThread;
    
    public boolean isMonitoring() {
        return (orderMonitorThread != null) && (orderMonitorThread.isAlive());
    };
    
    public boolean isDeamonMonitoring() {
        return (orderMonitorDeamonThread != null) && (orderMonitorDeamonThread.isAlive());
    };

    /**
     * The constructor need a data interface which he will monitore the orders.
     * The thread is launched at start and monitoring starts.
     * 
     * @param data
     */
    public VlOrderMonitoring(IVlData data) {
        this.data = data;
        // this.streamWatcher = new VlExtremeWatcher();
        // startMonitoringThread();
    }

    /**
     * Time interval between two RRD interrogations.
     */
    private static final int RRD_INTERVAL = 60;
    
    /**
     * The object which will give the multicast streams' list.
     */
    private IVlStreamWatcher streamWatcher;


    private Runnable orderMonitor = new Runnable() {
        public void run() {
            // Get the flow state from the router
            Map<InetAddress, InetAddress> streams = streamWatcher.getStreams();
            boolean shouldCompute = false;
            // Assigning programs state
            List<IVlMedia> medias = data.getMedias();
            for (IVlMedia media : medias) {
                IVlProgram program = media.getProgram();
                if (program != null) {
                    /*
                     * If the program must be played and is not then we have to
                     * recompute.
                     */
                    if (program.isTimeToPlay()) {
                        if (streams.get(program.getIp()) == null && !data.getOrderGiver().isComputing()) {
                            /*
                             * If the OrderGiver is not computing then it can
                             * modify the data.
                             */
                            program.setPlayer(null);
                            program.setBroadcastState(false);
                            shouldCompute = true;
                            /* Remove the order of a non-diffused stream */
                            Iterator<List<VlOrder>> k = data.getOrders().values().iterator();
                            while (k.hasNext()) {
                                Iterator<VlOrder> l = k.next().iterator();
                                while (l.hasNext()) {
                                    VlOrder o = l.next();
                                    if (o.getMedias().medias.contains(media)) {
                                        l.remove();
                                        break;
                                    }
                                }
                            }
                        }
                        else {
                            program.setBroadcastState(true);
                        }
                    }
                }
            }
            /*
             * Check if every media which belongs to a order has really a
             * program. If not, then it should compute.
             */
            for (List<VlOrder> orderList : data.getOrders().values()) {
                for (VlOrder order : orderList) {
                    for (IVlMedia media : order.getMedias().medias) {
                        if (media.getProgram() == null) {
                            logger.log(Level.DEBUG, "The media " + media.getName()
                                    + " which belongs to an order has no more a program.");
                            shouldCompute = true;
                            break;
                        }
                    }
                    if (shouldCompute == true)
                        break;
                }
                if (shouldCompute == true)
                    break;
            }
            
            /*
             * Something is wrong ! Ask for a new computation.
             */
            if (shouldCompute) {
                logger.log(Level.DEBUG, "Ask for programs reassignment.");
                data.giveOrders();
            }
        }
    };
    
    
    public synchronized void startOrderMonitoringThread() {
        if (!isMonitoring()) {
            orderMonitorThread = new Thread(orderMonitor);
            orderMonitorThread.setName("OrderMonitorThread");
            orderMonitorThread.start();
        }
    };
    
    private Runnable orderMonitorDeamon = new Runnable() {
        public void run() {
            while (true) {
                startOrderMonitoringThread();
                // Wait before looping
                try {
                    Thread.sleep(1000 * (RRD_INTERVAL - (Util.getTime() % RRD_INTERVAL)));
                } catch (InterruptedException e) {
                    
                }
            }
        }
    };
    

    public synchronized void startOrderMonitoringDeamon() {
        if (!isDeamonMonitoring()) {
            orderMonitorDeamonThread = new Thread(orderMonitorDeamon);
            orderMonitorDeamonThread.setName("OrderMonitorDeamonThread");
            orderMonitorDeamonThread.start();
        }
    };


    public void setStreamWatcher(IVlStreamWatcher streamWatcher) {
        this.streamWatcher = streamWatcher;
    };
};
