/*******************************************************************************
* dvbinput.cpp: DVB streams
*-------------------------------------------------------------------------------
* (c)1999-2002 VideoLAN
* $Id: dvbinput.cpp,v 1.5 2002/04/02 15:22:08 bozo Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../../core/defs.h"
#include "../../core/core.h"

#include <fcntl.h>
#include <sys/ioctl.h>

#include <dvbpsi/dvbpsi.h>
#include <dvbpsi/descriptor.h>
#include <dvbpsi/pat.h>
#include <dvbpsi/pmt.h>

#include <ost/sec.h>
#include <ost/frontend.h>
#include <ost/dmx.h>

#include "../../mpeg/mpeg.h"
#include "../../mpeg/ts.h"

#include "../../server/program.h"
#include "../../server/buffer.h"
#include "../../server/output.h"
#include "../../server/channel.h"
#include "../../server/broadcast.h"
#include "../../server/request.h"
#include "../../server/input.h"
#include "../../server/tsstreamer.h"

#include "../../mpeg/reader.h"
#include "../../mpeg/converter.h"
#include "../../mpeg/tsdemux.h"
#include "../../mpeg/dvbpsi.h"
#include "../../mpeg/tsmux.h"

#include "dvbinput.h"


//------------------------------------------------------------------------------
// Library declaration
//------------------------------------------------------------------------------
#ifdef __PLUGIN__
GENERATE_LIB_ARGS(C_DvbInputModule, handle);
#endif


//------------------------------------------------------------------------------
// Builtin declaration
//------------------------------------------------------------------------------
#ifdef __BUILTIN__
C_Module* NewBuiltin_dvbinput(handle hLog)
{
  return new C_DvbInputModule(hLog);
}
#endif


/*******************************************************************************
* C_DvbInput class
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_DvbInput::C_DvbInput(C_Module* pModule,
                       const C_String& strName) :
                                C_Input(pModule, strName),
                                C_TsDemux(&m_cTsProvider),
                                m_cTsProvider(500),
                                m_cInputProgram(0, "Input DVB " + strName),
                                m_cInputBroadcast(&m_cInputProgram, this, NULL),
                                m_cPatDecoder(&m_cTsProvider, this),
                                m_cCurrentPat(0, 0, true)
{
  m_pConverter = NULL;
  for(int i =0; i < 64; i++)
    m_iDemuxes[i] = -1;
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_DvbInput::~C_DvbInput()
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_DvbInput::OnInit()
{
  // Retrieve config
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);

  C_String strNumber = pApp->GetSetting(GetName() + "DeviceNumber", "");

  m_strSec      = "/dev/ost/sec" + strNumber;
  m_strFrontend = "/dev/ost/frontend" + strNumber;
  m_strDemux    = "/dev/ost/demux" + strNumber;
  m_strDvr      = "/dev/ost/dvr" + strNumber;

  m_iFrequency = pApp->GetSetting(GetName() + ".Frequency", "0").ToInt();
  m_iPolarization = pApp->GetSetting(GetName() + ".Polarization", "0").ToInt();
  m_iSymbolRate = pApp->GetSetting(GetName() + ".SymbolRate", "0").ToInt();

  m_iDiSEqC = pApp->GetSetting(GetName() + ".DiSEqC", "0").ToInt();

  m_iLnbLof1 = pApp->GetSetting(GetName() + ".LNB_Lof1", "0").ToInt();
  m_iLnbLof2 = pApp->GetSetting(GetName() + ".LNB_Lof2", "0").ToInt();
  m_iLnbSLof = pApp->GetSetting(GetName() + ".LNB_SLof", "0").ToInt();

  // Hardware initialization
  SecControl();
  SetFrontend();

  // PAT decoder initialization
  m_cPatDecoder.Attach();

  // Select the PAT PID
  SelectPid(&m_cPatDecoder, 0x0000, TS_TYPE_NULL);

  // Reader
  C_MpegReaderModule* pReaderModule = (C_MpegReaderModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegreader",
                                                    "file");
  ASSERT(pReaderModule);

  m_cInputBroadcast.SetOption("filename", m_strDvr);
  m_pReader = pReaderModule->NewMpegReader(&m_cInputBroadcast);
  ASSERT(m_pReader);

  // Converter
  C_MpegConverterModule* pConverterModule = (C_MpegConverterModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegconverter",
                                                    "ts2ts");
  ASSERT(pConverterModule);

  C_MpegConverterConfig cConfig;
  cConfig.m_hLog = m_hLog;
  cConfig.m_pBroadcast = &m_cInputBroadcast;
  cConfig.m_pReader = m_pReader;
  cConfig.m_pTsProvider = m_pTsProvider;
  cConfig.m_pHandler = this;
  cConfig.m_iInitFill = 0;
  cConfig.m_pEventHandler = this;
  m_pConverter = pConverterModule->NewMpegConverter(cConfig);
  ASSERT(m_pConverter);

  m_cEndInit.Protect();

  // Launch the demux
  m_pConverter->Create();

  // Kludge: wait for the first PAT arrival.
  m_cEndInit.Wait();
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
void C_DvbInput::OnDestroy()
{
  // Unselect the PAT PID
  C_TsDemux::Lock();
  UnselectPid(&m_cPatDecoder, 0x0000);
  C_TsDemux::UnLock();

  // PAT Decoder destruction
  m_cPatDecoder.Detach();

  if(m_pConverter)
  {
    // Stop the input stream
    try
    {
      m_pConverter->Stop();
    }
    catch(E_Exception e)
    {
      m_cEndInit.Release();
      delete m_pConverter;
      throw e;
    }

    delete m_pConverter;
  }

  m_cEndInit.Release();
}


//------------------------------------------------------------------------------
// Get the program table
//------------------------------------------------------------------------------
C_List<C_Program> C_DvbInput::OnGetAvailablePgrms()
{
  C_List<C_Program> cPgrmList;

  m_cLock.Lock();

  dvbpsi_pat_program_t *pPatProgram =
                        m_cCurrentPat.GetLowLevelPat()->p_first_program;
  while(pPatProgram)
  {
    C_Program* pProgram = new C_Program(pPatProgram->i_number,
                                        pPatProgram->i_number);
    ASSERT(pProgram);
    cPgrmList.PushEnd(pProgram);
    pPatProgram = pPatProgram->p_next;
  }

  m_cLock.UnLock();

  return cPgrmList;
}


//------------------------------------------------------------------------------
// Hardware PID selection
//------------------------------------------------------------------------------
void C_DvbInput::OnSelectPid(u16 iPid, u8 iType)
{
  int i;
  for(i = 0; (i < 32) && (m_iDemuxes[2 * i] != -1); i++);

  if(i < 32)
  {
    int iFd = open(m_strDemux.GetString(), O_RDWR|O_NONBLOCK);
    if(iFd < 0)
      Log(m_hLog, LOG_ERROR, "Unable to open demux");
    else
    {
      struct dmxPesFilterParams pesFilterParams;

      pesFilterParams.pid     = iPid;
      pesFilterParams.input   = DMX_IN_FRONTEND;
      pesFilterParams.output  = DMX_OUT_TS_TAP;
      switch(iType)
      {
      case TS_TYPE_MPEG1_VIDEO:
      case TS_TYPE_MPEG2_VIDEO:
        pesFilterParams.pesType = DMX_PES_VIDEO;
        break;
      case TS_TYPE_MPEG1_AUDIO:
      case TS_TYPE_MPEG2_AUDIO:
        pesFilterParams.pesType = DMX_PES_AUDIO;
        break;
      default:
        pesFilterParams.pesType = DMX_PES_OTHER;
        break;
      }
      pesFilterParams.flags  = DMX_IMMEDIATE_START;

      if(ioctl(iFd, DMX_SET_PES_FILTER, &pesFilterParams) < 0)
      {
        Log(m_hLog, LOG_ERROR, C_String("Unable to set demux filter for PID ") + iPid);
        close(iFd);
      }
      else
      {
        LogDbg(m_hLog, C_String("Demux filter set for PID ") + iPid);
        m_iDemuxes[2 * i] = iPid;
        m_iDemuxes[2 * i + 1] = iFd;
      }
    }
  }
  else
  {
    Log(m_hLog, LOG_ERROR, "To many PID selected");
  }
}


//------------------------------------------------------------------------------
// Harware PID unselection
//------------------------------------------------------------------------------
void C_DvbInput::OnUnselectPid(u16 iPid)
{
  int i;
  for(i = 0; (i < 32) && (m_iDemuxes[2 * i] != iPid); i++);

  if(i < 32)
  {
    ASSERT(m_iDemuxes[2 * i] == iPid);
    close(m_iDemuxes[2 * i + 1]);
    m_iDemuxes[2 * i] = -1;
    LogDbg(m_hLog, C_String("Demux filter unset for PID ") + iPid);
  }
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_DvbInput::HandleEvent(const C_Event& cEvent)
{
  ASSERT(false);
}


//------------------------------------------------------------------------------
// New version of the PAT
//------------------------------------------------------------------------------
void C_DvbInput::OnDvbPsiPatEvent(int iEvent)
{

  if(iEvent == DVBPSI_EVENT_CURRENT)
  {
    dvbpsi_pat_t *pLLPat = m_pCurrentPat->GetLowLevelPat();

    dvbpsi_pat_program_t* p_program = pLLPat->p_first_program;
    printf(  "\n");
    printf(  "New PAT\n");
    printf(  "  transport_stream_id : %d\n", pLLPat->i_ts_id);
    printf(  "  version_number      : %d\n", pLLPat->i_version);
    printf(  "    | program_number @ [NIT|PMT]_PID\n");
    while(p_program)
    {
      printf("    | %14d @ 0x%x (%d)\n",
             p_program->i_number, p_program->i_pid, p_program->i_pid);
      p_program = p_program->p_next;
    }
    printf(  "  active              : %d\n", pLLPat->b_current_next);

    C_DvbPsiPat DiffPatSub(0, 0, false);
    C_DvbPsiPat DiffPatAdd(0, 0, false);
    if(m_pPreviousPat)
    {
      DiffPatSub = *m_pPreviousPat - *m_pCurrentPat;
      DiffPatAdd = *m_pCurrentPat - *m_pPreviousPat;
    }
    else
    {
      DiffPatAdd = *m_pCurrentPat;
    }

    pLLPat = DiffPatSub.GetLowLevelPat();
    p_program = pLLPat->p_first_program;
    printf(  "\n");
    printf(  "Deleted programs\n");
    while(p_program)
    {
      printf("    | %14d @ 0x%x (%d)\n",
             p_program->i_number, p_program->i_pid, p_program->i_pid);
      p_program = p_program->p_next;
    }
    pLLPat = DiffPatAdd.GetLowLevelPat();
    p_program = pLLPat->p_first_program;
    printf(  "\n");
    printf(  "Added programs\n");
    while(p_program)
    {
      printf("    | %14d @ 0x%x (%d)\n",
             p_program->i_number, p_program->i_pid, p_program->i_pid);
      p_program = p_program->p_next;
    }

    m_cLock.Lock();
    m_cCurrentPat = *m_pCurrentPat;
    m_cLock.UnLock();

    // Kludge: signal the first PAT arrival.
    m_cEndInit.Protect();
    m_cEndInit.Signal();
  }
}


//------------------------------------------------------------------------------
// Hardware initialization
//------------------------------------------------------------------------------
void C_DvbInput::SecControl()
{
  struct secCommand sSCmd;
  struct secCmdSequence sSCmdSeq;
  int iSecFd;

  // Open the device
  iSecFd = open(m_strSec.GetString(), O_RDWR);

  if(iSecFd < 0)
  {
    throw E_Exception(GEN_ERR, "Unable to open \"" + m_strSec +
                      "\": " + GetErrorMsg());
  }

  // Set the frequency of the transponder, taking into account the
  // local frequencies of the LNB
  sSCmdSeq.continuousTone = (m_iFrequency < m_iLnbSLof) ?
                            SEC_TONE_OFF : SEC_TONE_ON;

  // Set the polarity of the transponder by setting the correct
  // voltage on the universal LNB
  sSCmdSeq.voltage = (m_iPolarization) ? SEC_VOLTAGE_18 : SEC_VOLTAGE_13;

  // In case we have a DiSEqC, set it to the correct address
  sSCmd.type = 0;
  sSCmd.u.diseqc.addr = 0x10;
  sSCmd.u.diseqc.cmd = 0x38;
  sSCmd.u.diseqc.numParams = 1;
  sSCmd.u.diseqc.params[0] =   0xF0
                            | ((m_iDiSEqC * 4) & 0x0F)
                            | (sSCmdSeq.continuousTone == SEC_TONE_ON ? 1 : 0)
                            | (sSCmdSeq.voltage == SEC_VOLTAGE_18 ? 2 : 0);

  sSCmdSeq.miniCommand = SEC_MINI_NONE;
  sSCmdSeq.numCommands = 1;
  sSCmdSeq.commands = &sSCmd;

  // Send the data to the SEC device to prepare the LNB for tuning
  Log(m_hLog, LOG_NOTE, "Sending data to the SEC");

  if(ioctl(iSecFd, SEC_SEND_SEQUENCE, &sSCmdSeq) < 0)
  {
    close(iSecFd);
    throw E_Exception(GEN_ERR, "Unable to send data to the SEC");
  }

  Log(m_hLog, LOG_NOTE, "Data successfully sent to the SEC\n");

  close(iSecFd);
}


//------------------------------------------------------------------------------
// Hardware initialization
//------------------------------------------------------------------------------
void C_DvbInput::SetFrontend()
{
  FrontendParameters sFeParams;
  int iFrontFd;

  // Open the device
  iFrontFd = open(m_strFrontend.GetString(), O_RDWR);

  if(iFrontFd < 0)
  {
    throw E_Exception(GEN_ERR, "Unable to open \"" + m_strSec +
                      "\": " + GetErrorMsg());
  }

  // Set the frequency of the transponder, taking into account the
  // local frequencies of the LNB
  sFeParams.Frequency = (m_iFrequency < m_iLnbSLof) ?
                        m_iFrequency - m_iLnbLof1 : m_iFrequency - m_iLnbLof2;

  // Set symbol rate and FEC
  sFeParams.u.qpsk.SymbolRate = m_iSymbolRate;
  sFeParams.u.qpsk.FEC_inner = FEC_AUTO;

  // Now send it all to the frontend device

  Log(m_hLog, LOG_NOTE, "Sending data to the Frontend");

  if(ioctl(iFrontFd, FE_SET_FRONTEND, &sFeParams) < 0)
  {
    close(iFrontFd);
    throw E_Exception(GEN_ERR, "Unable to send data to the Frontend");
  }

  Log(m_hLog, LOG_NOTE, "Data successfully sent to the Frontend\n");

  // Check if it worked
  fd_set sFdSet;
  struct timeval sTimeVal;
  FrontendEvent event;

  // poll for Frontend event to check if tuning worked
  FD_ZERO(&sFdSet);
  FD_SET(iFrontFd, &sFdSet);
  sTimeVal.tv_sec = 3;
  sTimeVal.tv_usec = 0;

  if(select(iFrontFd + 1, &sFdSet, NULL, NULL, &sTimeVal) == 1)
  {
    ASSERT(FD_ISSET(iFrontFd, &sFdSet));

    Log(m_hLog, LOG_NOTE, "Frontend: getting event");
    if(ioctl(iFrontFd, FE_GET_EVENT, &event) == -EBUFFEROVERFLOW)
    {
      close(iFrontFd);
      throw E_Exception(GEN_ERR, "Frontend: unable to get event");
    }

    Log(m_hLog, LOG_NOTE, "Frontend: event received");
    switch(event.type)
    {  
    case FE_UNEXPECTED_EV:
      close(iFrontFd);
      throw E_Exception(GEN_ERR, "Frontend: unexpected event");
    case FE_FAILURE_EV:
      close(iFrontFd);
      throw E_Exception(GEN_ERR, "Frontend: failure event");
    case FE_COMPLETION_EV:
      Log(m_hLog, LOG_NOTE, "Frontend: completion is OK");
    }
  }
  else
  {
    close(iFrontFd);
    throw E_Exception(GEN_ERR, "Frontend: time out: " + GetErrorMsg());
  }

  close(iFrontFd);
}


//------------------------------------------------------------------------------
// Start the reception of the given program
//------------------------------------------------------------------------------
void C_DvbInput::OnStartStreaming(C_Broadcast* pBroadcast)
{
  m_cLock.Lock();

  dvbpsi_pat_program_t *pProgram =
        m_cCurrentPat.GetProgram(pBroadcast->GetProgram()->GetName().ToInt());

  if(pProgram)
  {
    C_SyncFifo* pBuffer =
        new C_SyncFifo(2 * pBroadcast->GetChannel()->GetBuffCapacity());

    C_TsStreamer *pStreamer = new C_TsStreamer(m_hLog, pBroadcast,
                                               m_pTsProvider, pBuffer,
                                               m_pEventHandler, false, false);

    C_TsMux *pMux = new C_TsMux(m_pTsProvider, this, pBuffer);

    try
    {
      pStreamer->Create();

      pMux->Attach();

      pMux->AttachProgram(pProgram->i_number, pProgram->i_pid);

      m_cMuxes.Add(pProgram->i_number, pMux);
      m_cStreamers.Add(pProgram->i_number, pStreamer);
    }
    catch(E_Exception e)
    {
      delete pStreamer;
      delete pMux;
      throw e;
    }
  }
  else
  {
    throw E_Exception(GEN_ERR, "Program \"" +
                      pBroadcast->GetProgram()->GetName() +
                      "\" doesn't exist");
  }

  m_cLock.UnLock();
}


//------------------------------------------------------------------------------
// Resume the reception of the given program
//------------------------------------------------------------------------------
void C_DvbInput::OnResumeStreaming(C_Broadcast* pBroadcast)
{
}


//------------------------------------------------------------------------------
// Suspend the reception of the given program
//------------------------------------------------------------------------------
void C_DvbInput::OnSuspendStreaming(C_Broadcast* pBroadcast)
{
}


//------------------------------------------------------------------------------
// Stop the reception of the given program
//------------------------------------------------------------------------------
void C_DvbInput::OnStopStreaming(C_Broadcast* pBroadcast)
{
  m_cLock.Lock();

  u16 iNumber = pBroadcast->GetProgram()->GetName().ToInt();

  C_TsMux *pMux = m_cMuxes.Remove(iNumber);
  ASSERT(pMux);
  C_TsStreamer *pStreamer = m_cStreamers.Remove(iNumber);
  ASSERT(pStreamer);

  m_cLock.UnLock();

  pMux->Detach();
  delete pMux;

  try
  {
    pStreamer->Stop();
  }
  catch(E_Exception e)
  {
    delete pStreamer;
    throw E_Exception(GEN_ERR, "Unable to stop streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }

  delete pStreamer;
}


